# -*- coding: utf-8 -*-
import os, sys
import h5py, time
import numpy as np
import healpy as hp
from optparse import OptionParser
os.environ['NLS_LANG'] = 'SIMPLIFIED CHINESE_CHINA.UTF8'
os.environ['PYTHONIOENCODING'] = 'utf_8'
sys.path.append(os.path.dirname(__file__))



def SSM( **kwargs ) : 
	'''
	Parameters
	----------
	freq:
		unit [MHz], and must be `int/float`, NOT np.array/list/...

	nside: 
		`int`, nside=2**n
		nside of the output healpix map

	catalog:
		True | False
		=True: also return source catalog, Nx3 array, first column is RA [degree], second column is Dec [degree], third column is flux density [mJy]

	Smin: 
		`int/float`, [mJy]
		minimun flux density of the return source map

	name: 
		str, name of the output file
		if not give, default use 'ssm_CurrentTime.hdf5'

	verbose: 
		True | False
		=True: show the running messages

	inSSMpath:
		set the path of the 'inSSM.hdf5' file, if can NOT find it automatically
	'''
	# Current time
	nowsec1970 = time.localtime(time.time())
	createdate = time.strftime('%Y/%m/%d %H:%M:%S', nowsec1970)
	filedate   = time.strftime('%Y%m%d_%H%M%S', nowsec1970)
	key = list(kwargs.keys())

	# Check nside
	nside = kwargs.get('nside', 512)
	hp.nside2resol(nside)

	# catalog
	catalog = kwargs.get('catalog', False)
	# verbose
	verbose = kwargs.get('verbose', True)
	# Smin
	Smin = kwargs.get('Smin', None)

	# name, outname
	outname = kwargs.get('name', None)
	if (outname is None): 
		outname = 'ssm_'+filedate+'.hdf5'
	fmtname = outname[outname.rfind('.'):]
	if (fmtname.lower() not in ['.hdf5', '.h5']): 
		fmtname = '.hdf5'
		outname += fmtname

	# freq
	freq = kwargs.get('freq', None)
	if (freq is None) : 
		print('ERROR: missing parameter `freq`: freq must be int/float in MHz')
		exit()
	freq = np.array(freq)
	if (freq.shape != ()) : 
		print('ERROR: bad parameter `freq`: freq must be int/float in MHz, NOT list')
		exit()

	# inSSMpath
	inSSMpath = kwargs.get('inSSMpath', None)
	if (inSSMpath is None): 
		inSSMpath = os.path.expanduser(dest+'/inSSM.hdf5')

	if (verbose) : 
		print('Creating ...')
		print('    name    = %s' % outname)
		print('    freq    = %.4f MHz' % freq)
		print('    nside   = %i' % nside)
		print('    catalog = %s' % catalog)
		if (Smin is not None): 
			print('    Smin    = %.4f mJy' % Smin)
	#----------------------------------------

	def Index( freq ) : 
		freq = np.array(freq, float)
		shape = freq.shape
		freq = freq.flatten()
		beta = [2.54585438, 5.99296288e-02, 3.32505876e-02, 2.57045977e-03, -1.69742991e-03, -9.13846215e-05, 2.87044311e-05]
		index = beta[0]
		for i in range(1, len(beta)) : 
			index =index +beta[i]*(np.log(freq/408.))**i
		freq = np.log10(freq)
		kl, fl, il = 0.04790944268615008, 0.619600129210438, 2.4240227025232137
		tf = freq < fl
		index[tf] = kl*(freq[tf]-fl) + il
		kr, fr, ir = 0.00020614476389485252, 5.1578638486463495, 3.085344602580882
		tf = (freq > fr)
		index[tf] = kr*(freq[tf]-fr) + ir
		index = index.reshape(shape)
		return index

	def ToHealpix( nside, cat, replace ) : 
		RA, Dec, T = np.pi/180*cat[:,0], np.pi/180*cat[:,1], cat[:,2]
		if (replace): 
			T = np.sort(T + 1j*np.arange(T.size))
			n = T.imag.astype(int)
			T = T.real
			RA, Dec = RA[n], Dec[n]
		hpmap = np.zeros(12*nside**2)
		pix = hp.ang2pix(nside, np.pi/2-Dec, RA)
		if (replace): 
			hpmap[pix] = T
		else : 
			for i in range(pix.size): 
				hpmap[pix[i]] += T[i]
		return hpmap
	#----------------------------------------

	fo = h5py.File(inSSMpath, 'r')
	df, ds, dz, dr = fo['diffuse'].attrs['fszr']
	df = df[1:].astype(int)
#	dm = fo['diffuse'].value
	dm = fo['diffuse'][()]
	dm = np.sinh(dm * ds[:,None] + dz[:,None])
	ss, sz, sr = fo['source'].attrs['szr']
#	sc = fo['source'].value
	sc = fo['source'][()]
	sc = np.sinh(sc * ss[None,:] + sz[None,:]) + np.random.random(sc.shape)*2*sr[None,:]-sr[None,:]
	fo.close()
	#----------------------------------------

	sc = sc[sc[:,2]>0.3]
	sc[:,2] *= (1400./freq)**0.8157
	if (Smin is not None): 
		sc = sc[sc[:,2]>=Smin]
	else : 
		Smin = sc[:,2].min()
		if (verbose): print('    Smin    = %.4f mJy' % Smin)
	source = ToHealpix(nside, sc, False)
	#----------------------------------------

	if (freq < df[0]): 
		db = dm[1]
	elif (freq > df[-1]): 
		db = dm[-1]
	elif (abs(freq-int(freq)) <1e-6 and int(freq) in df) : 
		db = dm[np.where(df==int(freq))[0][0]+1]
	else : 
		for j in range(len(df)-1) : 
			if (df[j]>freq or freq>df[j+1]): continue
			f1, f2 = np.log10(df[j:j+2])
			f = np.log10(freq)
			d = (f2-f) / (f2-f1)
			db = dm[j+1]*d + dm[j+2]*(1-d)
	diffuse = dm[0]* (408./freq)**(db-db.mean()+Index(freq))
	diffuse = hp.ud_grade(diffuse, nside)
	#----------------------------------------

	if (os.path.exists(outname)): 
	#	os.system('mv '+outname+' '+outname[:-5]+'_old.hdf5')
		os.rename(outname, outname[:-len(fmtname)]+'_old'+fmtname)

	fo = h5py.File(outname, 'w')
	fo.attrs['comment'] = 'Data from SSM'
	fo.attrs['freq'] = freq
	fo.attrs['nside'] = nside
	fo.attrs['Smin'] = Smin 
	fo.attrs['name'] = outname
	fo.attrs['date'] = createdate
	fo.attrs['paper'] = 'Qizhi Huang et al. 2018'
	fo['diffuse'] = diffuse.astype('float32')
	fo['diffuse'].attrs['comment'] = 'Diffuse emission map'
	fo['diffuse'].attrs['freq'] = freq
	fo['diffuse'].attrs['nside'] = nside
	fo['diffuse'].attrs['unit'] = 'K'
	fo['diffuse'].attrs['coordsys'] = 'Galactic'
	fo['source'] = source.astype('float32')
	fo['source'].attrs['comment'] = 'Radio source map'
	fo['source'].attrs['freq'] = freq
	fo['source'].attrs['nside'] = nside
	fo['source'].attrs['unit'] = 'mJy'
	fo['source'].attrs['coordsys'] = 'Equatorial'
	if (catalog) : 
		fo['catalog'] = sc.astype('float32')
		fo['catalog'].attrs['comment'] = 'Radio source catalog'
		fo['catalog'].attrs['freq'] = freq
		fo['catalog'].attrs['Smin'] = Smin 
		fo['catalog'].attrs['unit'] = 'degree, degree, mJy'
		fo['diffuse'].attrs['coordsys'] = 'Equatorial'
	fo.close()
	return (diffuse, source, sc)





if (__name__ == '__main__') : 
	option = OptionParser('Also for python script:\n  from SSM import SSM\n  diffuse, source, catalog = SSM(freq=, nside=, catalog=, Smin=, name=, verbose=, inSSMpath=,)')
	option.add_option('--freq', action='store', type=float, dest='freq', help='[MHz], must be int/float, NOT list/...')
	option.add_option('--nside', action='store', type=int, dest='nside', default=512, help='nside=2^n, for the healpix')
	option.add_option('--catalog', action='store_true', dest='catalog', default=False, help='also save source catalog to .hdf5, Nx3 array, RA[deg],Dec[deg],S[mJy]')
	option.add_option('--Smin', action='store', type=float, dest='Smin', help='[mJy], minimun S of the source catalog')
	option.add_option('--name', action='store', type=str, dest='name', help='Name of the output .hdf5, default="ssm_%Y%m%d_%H%M%S.hdf5"')
	option.add_option('--notverbose', action='store_true', dest='notverbose', default=False, help="Don't show the running messages")
	option.add_option('--inSSMpath', action='store', type=str, dest='inSSMpath', help='Set the path of the `inSSM.hdf5` file, if can NOT find it automatically')
	option = option.parse_args()[0]
	if (option.inSSMpath is None) : option.inSSMpath = os.path.expanduser(dest+'/inSSM.hdf5')
	if (len(sys.argv) == 1) : 
		os.system('SSM -h')
		exit()
	SSM(freq=option.freq, nside=option.nside, catalog=option.catalog, Smin=option.Smin, name=option.name, verbose=not option.notverbose, inSSMpath=option.inSSMpath)

